/**
 * @class cDomainIntegerBasis
 * file name: cDomainIntegerBasis.h
 * @author Betti Oesterholz
 * @date 13.05.2009
 * @mail webmaster@BioKom.info
 *
 * System: C++
 *
 * This class represents a domain for integer numbers.
 * Copyright (C) @c LGPL3 2009 Betti Oesterholz
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (LGPL) as
 * published by the Free Software Foundation, either version 3 of the
 * License, or any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 *
 * This file contains the basisclass of all integer Fib -domains.
 * It's not possible to create instances from this class.
 *
 */
/*
History:
13.05.2009  Oesterholz  created
12.05.2010  Oesterholz  scalingfactor moved from cDomainSingle to cDomainIntegerBasis
16.05.2010  Oesterholz  getCompressedSizeForScalingFactor() added
*/

#ifndef ___C_DOMAIN_INTEGER_BASIS_H__
#define ___C_DOMAIN_INTEGER_BASIS_H__

#include "version.h"
#include "fibDatatyps.h"

#include "cDomainSingle.h"

#include <list>


namespace fib{


class cDomainIntegerBasis: public cDomainSingle{
protected:
	/**
	 * The scaling factor for the integer domains.
	 */
	doubleFib dScalingFactor;

	/**
	 * The constructor for scalar/ number domains.
	 *
	 * @param dInScalingFactor the scaling factor for the domain
	 * 	the scaling factor has to be greater 0.0
	 */
	cDomainIntegerBasis( const doubleFib dInScalingFactor );


public:

	/**
	 * This method returns the scalingfactor of the domain.
	 *
	 * @return the scalingfactor of the domain
	 */
	doubleFib getScalingFactor() const;

	/**
	 * This method returns the scaled value of the given value lValue.
	 *
	 * @param lValue the value to scale
	 * @return the scaled value of the given value lValue
	 */
	doubleFib scale( const longFib lValue ) const;


	/**
	 * This Method checks if the given domain is equal to this domain.
	 *
	 * @param domain the domain which should be equal to this domain
	 * @return true if the given domain is equal to this domain, else false
	 */
	virtual bool operator==( const cDomain &domain ) const;

	/**
	 * This method checks if the given value is an element of the
	 * unscaled domain. If the value dValue is an element of the
	 * unscaled domain true is returned, else false.
	 *
	 * @param dValue the value for which to check, if it is an element of
	 * 	the unscaled domain
	 * @return if the value dValue is an element of the unscaled
	 * 	domain true true is returned, else false
	 */
	virtual bool isUnscaledElement( const longFib dValue ) const = 0;


	/**
	 * This method round the given value dValue to an value in the
	 * unscaled domain of this object.
	 * The rounded vale will be the unscaled value with the minimal
	 * distance to the given value lValue. If more of these exists the 
	 * smales will be returned.
	 *
	 * @param lValue the value which should be rounded
	 * @return the rounded unscaled value of lValue
	 */
	virtual longFib roundUnscaled( const longFib lValue ) const = 0;


	/**
	 * This method returns the biggest value in the unscaled domain.
	 *
	 * @return the biggest value in the unscaled domain
	 */
	virtual longFib getMaximumUnscaled( ) const = 0;


	/**
	 * This method returns the smalest value in the unscaled domain.
	 *
	 * @return the smalest value in the unscaled domain
	 */
	virtual longFib getMinimumUnscaled( ) const = 0;


	/**
	 * This method returns the nullvalue of the unscaled domain.
	 * The nullvalue is the value 0 rounded to an value in the unscaled
	 * domain.
	 *
	 * @return the nullvalue of the domain
	 */
	virtual longFib getNullUnscaled( ) const = 0;


	/**
	 * This Method clones this object.
	 *
	 * @return a clone of this object
	 */
	virtual cDomainIntegerBasis *clone() const = 0;

	/**
	 * This method stores the given unscaled value lValue in the compressed
	 * fib -format for the domain into the given stream.
	 * It is needed because the stream can yust store byts but the size of
	 * fib -elements can be any number of bits. Because of that ther have to
	 * be a possibility to exchange the missing bits betwean the fib -elements.
	 *
	 * @see cFibElement::store
	 * @param lValue the value to store
	 * @param stream the stream where this domain should be stored to
	 * @param cRestBits the not yet writen bits which should be stored
	 * @param uiRestBitPosition the number of bits in the cRestBits which
	 * 	should be writen respectively containing valid information
	 * @return true if the domain is stored, else false
	 */
	virtual bool storeUnscaledValue( longFib lValue, ostream & stream,
		char & cRestBits, unsigned char & uiRestBitPosition ) const = 0;


	/**
	 * This method restores the a value of the domain from the stream
	 * iBitStream wher it is stored in the compressed fib -format.
	 *
	 * @see storeValue
	 * @see cFibElement::restore
	 * @param iBitStream the stream where this value is stored to in,
	 * 	because this stream is an cReadBits, any number of bits can be
	 * 	readed from it
	 * @param outStatus An reference to an integervalue where the errorvalue
	 * 	can be stored to. If the pointer is NULL no errorvalue will be
	 * 	given back.
	 * 	possible errorvalues are:
	 * 		- 0  loading successful
	 * 		- -1 loading error, invalid stream
	 * 		- -2 loading error, invalid data in stream
	 * 		- 1 loading warning, invalid data in stream, error could be corrected
	 * 		- 2 loading warning, invalid data in stream, maybe the loaded
	 * 			object is wrong
	 */
	virtual doubleFib restoreValue( cReadBits & iBitStream, intFib & outStatus ) const;

	/**
	 * This method restores the a unscaled integer value of the domain from
	 * the stream iBitStream, wher it is stored in the compressed fib -format.
	 *
	 * @see storeValue
	 * @see restoreValue
	 * @see storeUnscaledValue
	 * @see cFibElement::restore
	 * @param iBitStream the stream where this value is stored to in,
	 * 	because this stream is an cReadBits, any number of bits can be
	 * 	readed from it
	 * @param outStatus An reference to an integervalue where the errorvalue
	 * 	can be stored to. If the pointer is NULL no errorvalue will be
	 * 	given back.
	 * 	possible errorvalues are:
	 * 		- 0  loading successful
	 * 		- -1 loading error, invalid stream
	 * 		- -2 loading error, invalid data in stream
	 * 		- 1 loading warning, invalid data in stream, error could be corrected
	 * 		- 2 loading warning, invalid data in stream, maybe the loaded
	 * 			object is wrong
	 */
	virtual longFib restoreIntegerValue( cReadBits & iBitStream, intFib & outStatus ) const = 0;

	/**
	 * This function creates creates a good domain wich contains all the
	 * numbers of the given list liValues.
	 * A domain is better than an other domain, if it takes less storage bits
	 * to store the numbers and the domain.
	 * Beware: You have to care that the returned domain is deleted. (if
	 * 	one was returned)
	 *
	 * @param liValues the list with the values, for which a domain is to create
	 * @return a new domain which contains all the given values liValues
	 */
	static cDomainIntegerBasis * createGoodDomain( const list< longFib > & liValues );

protected:

	/**
	 * This method evaluades the size for the scaling factor in bits in the
	 * compressed file form.
	 *
	 * @see storeScalingFactor()
	 * @return the size of the domain in bits in the compressed form
	 */
	unsigned int getCompressedSizeForScalingFactor() const;
	
	/**
	 * This method stores the scalingfactor of this domain in the
	 * compressed fib -format into the given stream.
	 * It is needed because the stream can yust store byts but the size of
	 * fib -elements can be any number of bits. Because of that ther have to
	 * be a possibility to exchange the missing bits betwean the fib -elements.
	 *
	 * @see cFibElement::store
	 * @param stream the stream where this domain should be stored to
	 * @param cRestBits the not yet writen bits which should be stored
	 * @param uiRestBitPosition the number of bits in the cRestBits which
	 * 	should be writen respectively containing valid information
	 * @return true if the scalingfactor of the domain is stored, else false
	 */
	bool storeScalingFactor( ostream & stream, char & cRestBits,
		unsigned char & uiRestBitPosition ) const;

	/**
	 * This method restores the scalingfactore from a bitstream, wher it is
	 * stored in the compressed fib -format.
	 *
	 * @see storeScalingFactor
	 * @see restore
	 * @param iBitStream the stream where the scalingfactore is stored in,
	 * 	because the stream is an cReadBits, any number of bits can be
	 * 	readed from it
	 * @return an integervalue with the errorvalue
	 * 	possible errorvalues are:
	 * 		- 0  loading successful
	 * 		- -1 loading error, invalid pXmlElement
	 * 		- -2 loading error, invalid data in pXmlElement
	 * 		- 1 loading warning, invalid data in pXmlElement, error could be corrected
	 * 		- 2 loading warning, invalid data in pXmlElement, maybe the loaded
	 * 			object is wrong
	 */
	intFib restoreScalingFactor( cReadBits & iBitStream );

	/**
	 * This function sets the scalingfactor.
	 * The scalingfactor is given as the scalingfactor in xml -format.
	 *
	 * @param szXmlScalingFactor the scalingfactor readed from an fib -xml -format
	 */
	doubleFib setXmlScaling( const char * szXmlScalingFactor );


};//end class cDomainIntegerBasis


}//end namespace fib

#endif
